<?php

namespace App\Http\Controllers\Dashboard;

use App\Http\Controllers\Controller;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Str;
use Auth;

class UserController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        // Logic to list users
        $users = User::with('roles')->orderBy('id','desc')->where('is_super_admin', 0)->paginate(20);
        return view('panel.users.index', compact('users'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        // Logic to show user creation form
        return view('panel.users.create');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {

        // Logic to store a new user
        $validator = Validator::make($request->all(), [
            'name'     => 'required|string|max:255|unique:suppliers,name',
            'company_id' => 'required|exists:companies,id',
            'device_user_id' => 'required|string|max:255|unique:users,device_user_id',
            'email'    => 'required|string|email|max:255|unique:users,email',
            'phone'    => 'required|string|max:15|unique:users,phone',
            'address'  => 'nullable|string|max:500',
            'password' => 'required|string|min:8|confirmed',
        ]);

        if ($validator->fails()) {

            return redirect()->back()->withErrors($validator)->withInput();
        }

        $user = User::create([
            'name' => $request->name,
            'device_user_id' => $request->device_user_id,
            'shift_id' => $request->shift_id,
            'company_id' => $request->company_id,
            'email' => $request->email,
            'phone' => $request->phone,
            'address' => $request->address,
            'is_active' => $request->is_active ? 'active' : 'inactive',
            'password' => bcrypt($request->password),

        ]);

        $user->assignRole('employee');


        return redirect()->route('role.user.index', ['role' => Str::slug(Auth::user()->getRoleNames()->first())])->with('success', 'User created successfully.');
    }

    // Other methods like show, edit, update, destroy can be added similarly
    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        // Logic to show a specific user
        return view('panel.users.show', compact('id'));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($role, $id)
    {
        // Logic to show user edit form
        $data = User::findOrFail($id);
        return view('panel.users.edit', compact('data'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request,$role, $id)
    {
        // Logic to update a user
        $user = User::findOrFail($id);

        $validator = Validator::make($request->all(), [
            'name'     => 'required|string|max:255',
            'company_id' => 'required|exists:companies,id',
            'device_user_id' => 'required|string|max:255|unique:users,device_user_id,' . $user->id,
            'shift_id' => 'nullable|exists:shifts,id',
            'email'    => 'required|string|email|max:255|unique:users,email,' . $user->id,
            'phone'    => 'required|string|max:15|unique:users,phone,' . $user->id,
            'address'  => 'nullable|string|max:500',
        ]);

        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator)->withInput();
        }

        $user->update([
            'name' => $request->name,
            'device_user_id' => $request->device_user_id,
            'shift_id' => $request->shift_id,
            'company_id' => $request->company_id,
            'email' => $request->email,
            'phone' => $request->phone,
            'address' => $request->address,
        ]);
        if ($request->password) {
            $user->update([
                'password' => bcrypt($request->password),
            ]);
        }

        $user->syncRoles([$request->role]);

        // Validate and update user data here
        return redirect()->route('role.user.index', ['role' => $request->role])->with('success', 'User updated successfully.');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        // Logic to delete a user
        // Delete user data here
        return redirect()->route('role.user.index')->with('success', 'User deleted successfully.');
    }

}
