<?php

namespace App\Http\Controllers\Report;

use App\Exports\AttendanceExport;
use App\Http\Controllers\Controller;
use App\Models\Attendance;
use App\Models\User;
use Carbon\Carbon;
use Carbon\CarbonPeriod;
use Illuminate\Http\Request;
use Maatwebsite\Excel\Excel;

class ReportController extends Controller
{
    public function monthly_attendances(Request $request)
    {
        $users = User::whereNotNull('company_id')->get();
        $selectedMonth = $request->input('month', date('n'));
        $selectedYear = $request->input('year', date('Y'));
        $userId = $request->input('user_id');

        $fullMonthData = collect();
        // Adjusted summary to match the logic where Late is a subset of Present
        $summary = [
            'present' => 0,
            'absent' => 0,
            'leave' => 0,
            'holiday' => 0,
            'late' => 0,
            'total_days' => 0
        ];

        if ($userId) {
            $user = User::with('shift')->find($userId);

            $dbAttendances = Attendance::with('shift', 'attendence_setting')
                ->whereMonth('date', $selectedMonth)
                ->whereYear('date', $selectedYear)
                ->where('user_id', $userId)
                ->get()
                ->keyBy('date');

            $startDate = \Carbon\Carbon::create($selectedYear, $selectedMonth, 1)->startOfMonth();
            $endDate = $startDate->copy()->endOfMonth();
            $period = \Carbon\CarbonPeriod::create($startDate, $endDate);

            // Fallback shift from user if no attendance records exist yet
            $defaultShift = $user->shift ?? $dbAttendances->first()?->shift;

            // Fetch Holidays that overlap with the selected month
            $holidays = \DB::table('holidays')
                ->where(function ($query) use ($startDate, $endDate) {
                    $query->whereBetween('start_date', [$startDate, $endDate])
                        ->orWhereBetween('end_date', [$startDate, $endDate]);
                })->get();

            // Fetch Approved Leaves for this user
            $leaves = \DB::table('leaves')
                ->where('user_id', $userId)
                ->where('status', 'approved')
                ->where(function ($query) use ($startDate, $endDate) {
                    $query->whereBetween('start_date', [$startDate, $endDate])
                        ->orWhereBetween('end_date', [$startDate, $endDate]);
                })->get();

            foreach ($period as $date) {
                $dateString = $date->format('Y-m-d');
                $dayName = strtolower($date->format('l'));

                if (isset($dbAttendances[$dateString])) {
                    $record = $dbAttendances[$dateString];

                    // --- CONSISTENT LATE CALCULATION ---
                    if ($record->check_in && $record->attendence_setting) {
                        $checkInTime = \Carbon\Carbon::parse($dateString . ' ' . $record->check_in);
                        $lateThreshold = \Carbon\Carbon::parse($dateString . ' ' . $record->attendence_setting->time_after_checkin);

                        if ($checkInTime->gt($lateThreshold)) {
                            $record->status = 'Late';
                            $summary['late']++;    // Increment Late
                            $summary['present']++; // Also increment Present (as they are physically there)
                        } else {
                            $record->status = 'Present';
                            $summary['present']++;
                        }
                    } else {
                        $status = strtolower($record->status);
                        if (array_key_exists($status, $summary)) {
                            $summary[$status]++;
                        }
                    }
                }  elseif ($leaveRecord = $leaves->first(fn($l) => $dateString >= $l->start_date && $dateString <= $l->end_date)) {
                    $record = (object)[
                        'date' => $dateString,
                        'shift' => $defaultShift,
                        'check_in' => null,
                        'check_out' => null,
                        'status' => 'Leave',
                        'note' => 'Approved Leave'
                    ];
                    $summary['leave']++;
                } else {
                    $holidayRecord = $holidays->first(fn($h) => $dateString >= $h->start_date && $dateString <= $h->end_date);
                    $isWeekend = $defaultShift && in_array($dayName, $defaultShift->holidays ?? []);

                    if ($holidayRecord || $isWeekend) {
                        $record = (object)[
                            'date' => $dateString,
                            'shift' => $defaultShift,
                            'check_in' => null,
                            'check_out' => null,
                            'status' => 'Holiday',
                            'note' => $holidayRecord ? $holidayRecord->name : 'Weekend'
                        ];
                        $summary['holiday']++;
                    } else {
                        // 4. Default to Absent
                        $record = (object)[
                            'date' => $dateString,
                            'shift' => $defaultShift,
                            'check_in' => null,
                            'check_out' => null,
                            'status' => 'Absent',
                            'note' => 'No record'
                        ];
                        $summary['absent']++;
                    }
                }

                $fullMonthData->push($record);
            }
            $summary['total_days'] = $period->count();
        }

        return view('report.monthly_attendances', [
            'attendances' => $fullMonthData,
            'users' => $users,
            'summary' => $summary,
            'selectedMonth' => $selectedMonth,
            'selectedYear' => $selectedYear,
            'userId' => $userId
        ]);
    }

    public function monthlyAttendanceExcel(Request $request)
    {
        $month = $request->input('month', date('n'));
        $year = $request->input('year', date('Y'));
        $userId = $request->input('user_id');
        $user = User::with('shift')->find($userId);

        $dbAttendances = Attendance::with('shift', 'attendence_setting')
            ->whereMonth('date', $month)
            ->whereYear('date', $year)
            ->where('user_id', $userId)
            ->get()->keyBy('date');

        $fullMonthData = collect();
        $startDate = \Carbon\Carbon::create($year, $month, 1)->startOfMonth();
        $endDate = $startDate->copy()->endOfMonth();
        $period = \Carbon\CarbonPeriod::create($startDate, $endDate);

        $defaultShift = $user->shift ?? $dbAttendances->first()?->shift;

        // Fetch Holidays that overlap with the selected month
        $holidays = \DB::table('holidays')
            ->where(function ($query) use ($startDate, $endDate) {
                $query->whereBetween('start_date', [$startDate, $endDate])
                    ->orWhereBetween('end_date', [$startDate, $endDate]);
            })->get();

        // Fetch Approved Leaves for this user
        $leaves = \DB::table('leaves')
            ->where('user_id', $userId)
            ->where('status', 'approved')
            ->where(function ($query) use ($startDate, $endDate) {
                $query->whereBetween('start_date', [$startDate, $endDate])
                    ->orWhereBetween('end_date', [$startDate, $endDate]);
            })->get();

        // Initialize counters
        $totalMinutes = 0;
        $present = 0;
        $absent = 0;
        $late = 0;
        $holiday = 0;
        $leave = 0;        

        foreach ($period as $date) {
            $dateString = $date->format('Y-m-d');
            $dayName = strtolower($date->format('l'));

            if (isset($dbAttendances[$dateString])) {
                $record = $dbAttendances[$dateString];
                $record->shift_name = $record->shift?->name ?? '-';
                $record->worked_display = "Missing Out";

                if ($record->check_in && $record->check_out) {
                    $start2 = \Carbon\Carbon::parse($dateString . ' ' . $record->check_in);
                    $end2 = \Carbon\Carbon::parse($dateString . ' ' . $record->check_out);

                    // Special case: Overnight shift (checkout next day)
                    if ($end2->lt($start2)) {
                        $end2->addDay();
                    }

                    $diffInMinutes2 = $start2->diffInMinutes($end2);
                    $totalMinutes += $diffInMinutes2;

                    // Format for the individual Excel row: "8h 30m"
                    $h2 = floor($diffInMinutes2 / 60);
                    $m2 = $diffInMinutes2 % 60;
                    $record->worked_display = "{$h2}h {$m2}m";
                }

                // --- CONSISTENT LATE CALCULATION ---
                if ($record->check_in && $record->attendence_setting) {
                    // Prepend dateString so Carbon doesn't use "today"
                    $checkInTime = \Carbon\Carbon::parse($dateString . ' ' . $record->check_in);
                    $boundaryTimeStr = $record->attendence_setting->time_after_checkin; // e.g. "10:15"
                    $lateThreshold = \Carbon\Carbon::parse($dateString . ' ' . $boundaryTimeStr);

                    if ($checkInTime->gt($lateThreshold)) {
                        $record->status = 'Late';
                        $late++;
                        $present++; // Late people are physically present
                    } else {
                        $record->status = 'Present';
                        $present++;
                    }
                } else {
                    // If no check_in (e.g. manual status entries like Leave)
                    $currentStatus = strtolower($record->status);
                    if ($currentStatus == 'leave') $leave++;
                    elseif ($currentStatus == 'holiday') $holiday++;
                    else $absent++;
                }
            } elseif ($leaveRecord = $leaves->first(fn($l) => $dateString >= $l->start_date && $dateString <= $l->end_date)) {
                $record = (object)[
                    'date' => $dateString,
                    'employee_id' => $user->employee_id ?? '',
                    'shift_name' => $defaultShift?->name ?? '-',
                    'check_in' => null,
                    'check_out' => null,
                    'worked_display' => "--",
                    'status' => 'Leave',
                    'note' => 'Approved Leave'
                ];
                $leave++;
            } else {
                $holidayRecord = $holidays->first(fn($h) => $dateString >= $h->start_date && $dateString <= $h->end_date);
                $isWeekend = $defaultShift && in_array($dayName, $defaultShift->holidays ?? []);

                if ($holidayRecord || $isWeekend) {
                    $record = (object)[
                        'date' => $dateString,
                        'employee_id' => $user->employee_id ?? '',
                        'shift_name' => $defaultShift?->name ?? '-',
                        'check_in' => null,
                        'check_out' => null,
                        'worked_display' => "--",
                        'status' => 'Holiday',
                        'note' => $holidayRecord ? $holidayRecord->name : 'Weekend'
                    ];
                    $holiday++;
                } else {
                    // 4. Default to Absent
                    $record = (object)[
                        'date' => $dateString,
                        'employee_id' => $user->employee_id ?? '',
                        'shift_name' => $defaultShift?->name ?? '-',
                        'check_in' => null,
                        'check_out' => null,
                        'worked_display' => "--",
                        'status' => 'Absent',
                        'note' => 'No record'
                    ];
                    $absent++;
                }
            }
            $fullMonthData->push($record);
        }

        // --- APPEND SUMMARY ROWS ---
        // We put values in 'shift_name' so the Export's map() function finds them easily
        $summaryRow = function ($label, $value = '') {
            return (object)[
                'date' => $label,
                'shift_name' => $value,
                'check_in' => null,
                'check_out' => null,
                'worked_display' => "--",
                'status' => '',
                'note' => ''
            ];
        };

        $totalH = floor($totalMinutes / 60);
        $totalM = $totalMinutes % 60;
        $totalWorkedString = "{$totalH}H and {$totalM}M";

        $fullMonthData->push($summaryRow('')); // Spacer
        $fullMonthData->push($summaryRow('TOTAL SUMMARY'));
        $fullMonthData->push($summaryRow('Total Present:', $present ?? 0));
        $fullMonthData->push($summaryRow('Total Late:', $late ?? 0));
        $fullMonthData->push($summaryRow('Total Worked:', $totalWorkedString));
        $fullMonthData->push($summaryRow('Total Absent:', $absent ?? 0));
        $fullMonthData->push($summaryRow('Total Holiday:', $holiday ?? 0));
        $fullMonthData->push($summaryRow('Total Leave:', $leave ?? 0));

        return \Excel::download(new AttendanceExport($fullMonthData, $user), "Attendance_Report_{$userId}_{$month}_{$year}.xlsx");
    }

    public function monthlyAttendancePdf(Request $request)
    {
        $month = $request->input('month', date('n'));
        $year = $request->input('year', date('Y'));
        $userId = $request->input('user_id');
        $user = User::with('shift')->find($userId);

        // 1. Fetch Data (Same logic as Excel/Web)
        $dbAttendances = Attendance::with('shift', 'attendence_setting')
            ->whereMonth('date', $month)
            ->whereYear('date', $year)
            ->where('user_id', $userId)
            ->get()->keyBy('date');

        $startDate = \Carbon\Carbon::create($year, $month, 1)->startOfMonth();
        $endDate = $startDate->copy()->endOfMonth();
        $period = \Carbon\CarbonPeriod::create($startDate, $endDate);
        $defaultShift = $user->shift ?? $dbAttendances->first()?->shift;

        // Fetch Holidays that overlap with the selected month
        $holidays = \DB::table('holidays')
            ->where(function ($query) use ($startDate, $endDate) {
                $query->whereBetween('start_date', [$startDate, $endDate])
                    ->orWhereBetween('end_date', [$startDate, $endDate]);
            })->get();

        // Fetch Approved Leaves for this user
        $leaves = \DB::table('leaves')
            ->where('user_id', $userId)
            ->where('status', 'approved')
            ->where(function ($query) use ($startDate, $endDate) {
                $query->whereBetween('start_date', [$startDate, $endDate])
                    ->orWhereBetween('end_date', [$startDate, $endDate]);
            })->get();

        $fullMonthData = collect();
        $summary = ['present' => 0, 'absent' => 0, 'late' => 0, 'holiday' => 0, 'leave' => 0];

        foreach ($period as $date) {
            $dateString = $date->format('Y-m-d');            
            $dayName = strtolower($date->format('l'));

            if (isset($dbAttendances[$dateString])) {
                $record = $dbAttendances[$dateString];
                $record->shift_name = $defaultShift->name ?? '-';
                if ($record->check_in && $record->attendence_setting) {
                    $checkInTime = \Carbon\Carbon::parse($dateString . ' ' . $record->check_in);
                    $lateThreshold = \Carbon\Carbon::parse($dateString . ' ' . $record->attendence_setting->time_after_checkin);

                    if ($checkInTime->gt($lateThreshold)) {
                        $record->status = 'Late';
                        $summary['late']++;
                        $summary['present']++;
                    } else {
                        $record->status = 'Present';
                        $summary['present']++;
                    }
                } else {
                    $status = strtolower($record->status);
                    if (array_key_exists($status, $summary)) $summary[$status]++;
                }
            } elseif ($leaveRecord = $leaves->first(fn($l) => $dateString >= $l->start_date && $dateString <= $l->end_date)) {
                $record = (object)[
                    'date' => $dateString,
                    'shift_name' => $defaultShift->name ?? '-',
                    'check_in' => null,
                    'check_out' => null,
                    'status' => 'Leave',
                    'note' => 'Approved Leave'
                ];
                $summary['leave']++;
            } else {
                $holidayRecord = $holidays->first(fn($h) => $dateString >= $h->start_date && $dateString <= $h->end_date);
                $isWeekend = $defaultShift && in_array($dayName, $defaultShift->holidays ?? []);

                if ($holidayRecord || $isWeekend) {
                    $record = (object)[
                        'date' => $dateString,
                        'shift_name' => $defaultShift->name ?? '-',
                        'check_in' => null,
                        'check_out' => null,
                        'status' => 'Holiday',
                        'note' => $holidayRecord ? $holidayRecord->name : 'Weekend'
                    ];
                    $summary['holiday']++;
                } else {
                    // 4. Default to Absent
                    $record = (object)[
                        'date' => $dateString,
                        'shift_name' => $defaultShift->name ?? '-',
                        'check_in' => null,
                        'check_out' => null,
                        'status' => 'Absent',
                        'note' => 'No record'
                    ];
                    $summary['absent']++;
                }
            }
            $fullMonthData->push($record);
        }

        // 2. Generate PDF using a specific blade view
        $pdf = \PDF::loadView('report.monthly_attendances_pdf', [
            'attendances' => $fullMonthData,
            'user' => $user,
            'summary' => (object)$summary,
            'month' => $startDate->format('F'),
            'year' => $year
        ]);

        return $pdf->download("Attendance_Report_{$user->name}_{$month}.pdf");
    }

}
